/** @file   poly2d.h
 * @brief   Declaration and implementation of Polygon - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_POLYGON_H
#define H_ENG2D_POLYGON_H

#include <vector>
#include <allegro.h>
#include "eng2d_dll.h"
#include "vec2d.h"
#include "eng2d_Exception.h"

namespace eng2d {

/** @class  Poly2D
 * @brief   Represents a non graphical polygon with any number of vertices.
 * @author  Tomi Lamminsaari
 *
 * This class represents some kind of geometric Poly2D. You can set up
 * any number of vertices starting from 3.
 */
class DLLIMPORT Poly2D
{
protected:

  ///
  /// Members
  /// =======
  
  /** The vertices of this Poly2D are stored here. */
  std::vector< Vec2D > m_vertices;



public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor. Constructs new Poly2D with given number of vertices. By
   * default all the vertices are located in origo.
   * @param     vcount            Number of vertices. If less than 3, an
   *                              exception will be thrown.
   * @exception eng2d_Exception   Exception of this type will be thrown if
   *                              given parameter was invalid.
   */
	inline Poly2D( int vcount )
	{
    if ( vcount < 3 ) {
      throw xLogicError( "Poly2D", "Poly2D(int vcount)", "too few vertices" );
    }
    m_vertices.resize( vcount );
  }

  /** Constructor. Constructs a Poly2D with given vertices.
   * @param     v1                Vertex 1
   * @param     v2                Vertex 2
   * @param     v3                Vertex 3
   */
  inline Poly2D( const Vec2D& v1, const Vec2D& v2, const Vec2D& v3 )
  {
    m_vertices.push_back( v1 );
    m_vertices.push_back( v2 );
    m_vertices.push_back( v3 );
  }

	/** Destructor
   */
	virtual ~Poly2D()
	{
  }

	/** Copy constructor.
   * @param     rO                Reference to another Poly2D
   */
  inline Poly2D( const Poly2D& rO ) :
    m_vertices( rO.m_vertices )
  {
  }

	/** Assignment operator
   * @param     rO                Reference to another Poly2D
   * @return    Reference to us.
   */
  inline Poly2D& operator = ( const Poly2D& rO )
  {
    if ( this != &rO ) {
      m_vertices = rO.m_vertices;
    }
    return *this;
  }

  /** Equality operator
   */
  inline bool operator == ( const Poly2D& rP )
  {
    if ( this->vertexCount() != rP.vertexCount() ) {
      return false;
    }
    
    for ( int i=0; i < m_vertices.size(); i++ ) {
      if ( m_vertices.at(i) != rP.m_vertices.at(i) ) {
        return false;
      }
    }
    return true;
  }

  /** Inequality operator
   */
  inline bool operator != ( const Poly2D& rP )
  {
    return ! (this->operator==( rP ));
  }
  


  ///
  /// Methods
  /// =======

  /** Returns a reference to index'th vertex so that it can be modified.
   * The there is no vertex in given index, we throw an exception of
   * type @c std::exception
   * @param     index             Index of the vertex.
   * @return    A reference to the vertex so that it can be changed.
   */
  inline Vec2D& vtx( int index )
  {
    return m_vertices.at( index );
  }

  /** Adds new vertex to this polygon. The new vertex will go to the
   * end of vertex list.
   * @param     rV                Position of the vertex.
   * @return    Index of the vertex we just added.
   */
  inline int addVtx( const Vec2D& rV )
  {
    m_vertices.push_back( rV );
    return this->vertexCount() - 1;
  }

  

  ///
  /// Getter methods
  /// ==============

  /** Returns a the index'th vertex. If there is no vertex in given index
   * we throw an exception of type @c std::exception
   * @param     index             Index of the refence.
   * @return    The vertex coordinate.
   */
  inline const Vec2D& vtx( int index ) const
  {
    return m_vertices.at( index );
  }
  
  /** Returns the number of vertices there are.
   * @return    Number of vertices.
   */
  inline int vertexCount() const
  {
    return m_vertices.size();
  }
  
  /** Tells if given point is inside this polygon. NOT IMPLEMENTED !
   * @param     rP                The refence point
   * @return    <code>true</code> if the point is inside this polygon.
   */
  inline bool inside( const Vec2D& rP ) const
  {
    return false;
  }
  


private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
